/**
  * # AWS - Amazon Elasticsearch Service
  * 
  */
locals {
  es_domain_name = (var.es_domain_name_prefix_override != "" ? "${var.es_domain_name_prefix_override}" : "${var.full_workload}-es")
}

data "aws_region" "current" {}

data "aws_caller_identity" "current" {}

data "aws_availability_zones" "available" {
  state = "available"
}

resource "aws_iam_service_linked_role" "es" {
  aws_service_name = "es.amazonaws.com"
}

resource "aws_security_group" "es" {
  count = var.es_enabled ? 1 : 0

  name        = "${local.es_domain_name}-sg"
  description = "Allow inbound traffic to ElasticSearch from VPC CIDR"
  vpc_id      = var.vpc_id

  ingress {
    from_port = 0
    to_port   = 0
    protocol  = "-1"
    cidr_blocks = [
      var.vpc_cidr_block
    ]
  }
}

resource "aws_elasticsearch_domain" "es" {
  count = var.es_enabled ? 1 : 0

  domain_name           = local.es_domain_name
  elasticsearch_version = "7.10"

  cluster_config {
    instance_count         = var.es_cluster_instance_count
    instance_type          = var.es_cluster_instance_type
    zone_awareness_enabled = true

    zone_awareness_config {
      availability_zone_count = 3
    }
  }

  vpc_options {
    subnet_ids = var.vpc_private_subnets

    security_group_ids = [
      aws_security_group.es[0].id
    ]
  }

  ebs_options {
    ebs_enabled = true
    volume_size = 10
  }

  access_policies = <<CONFIG
{
  "Version": "2012-10-17",
  "Statement": [
      {
          "Action": "es:*",
          "Principal": "*",
          "Effect": "Allow",
          "Resource": "arn:aws:es:${data.aws_region.current.name}:${data.aws_caller_identity.current.account_id}:domain/${local.es_domain_name}/*"
      }
  ]
}
  CONFIG

  snapshot_options {
    automated_snapshot_start_hour = 23
  }

  tags = {
    Domain = "${local.es_domain_name}"
  }

  depends_on = [aws_iam_service_linked_role.es]
}
